from django.db import models
from subscriptions.models import Profile
import uuid
import enum
from lib.choices import choices_the_options


class ApprovalStatus(enum.Enum):
    OPEN = 'open'
    APPROVED = 'approved'
    NOT_APPROVED = 'not_approved'

class DailyReport(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    profiles = models.ForeignKey(Profile, on_delete=models.PROTECT, related_name='daily_reports')
    date = models.DateField()
    start_tasks = models.TextField(blank=True, null=True)
    job_order_no = models.CharField(max_length=255, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.date} - {self.job_order_no}"
    
    class Meta:
        ordering = ['-created_at']

class OfficeExpense(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    date = models.DateField(auto_now_add=True)
    expence = models.TextField(blank=True, null=True)
    debit = models.DecimalField(max_digits=10, decimal_places=2, blank=True, null=True)
    credit = models.DecimalField(max_digits=10, decimal_places=2, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.date} - {self.expence or 'Expense'}"
    
    class Meta:
        ordering = ['-created_at']

class CompanyCalendar(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    client_name = models.CharField(max_length=255, null=True, blank=True)
    nature_of_business = models.CharField(max_length=255, null=True, blank=True)
    handled_by = models.ForeignKey(Profile, on_delete=models.CASCADE, related_name='handled_calendars')
    business_type = models.CharField(max_length=255, null=True, blank=True)
    responsible_person = models.CharField(max_length=255, null=True, blank=True)
    
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)  

    class Meta:
        ordering = ['-created_at']

    def __str__(self):
        return self.client_name or "Company Calendar"

class ContentCalendar(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    company = models.ForeignKey(CompanyCalendar, on_delete=models.PROTECT, related_name='content_calendars')
    activities = models.CharField(max_length=255, null=True, blank=True)
    content = models.TextField()
    sales_order_no  = models.CharField(max_length=255, null=True, blank=True)
    target_date  = models.DateField(null=True, blank=True)
    completion_date  = models.DateField(null=True, blank=True)
    approved = models.CharField(max_length=255, choices=choices_the_options(ApprovalStatus), default=ApprovalStatus.OPEN.value)
    remarks = models.TextField(null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    # class Meta:
    #     ordering = ['-created_at']

    def __str__(self):
        return f"{self.company.client_name} - {self.target_date} to {self.completion_date}"
    
    def clean(self):
        from django.core.exceptions import ValidationError

        if self.target_date and self.completion_date and self.target_date > self.completion_date:
            raise ValidationError({'completion_date': "Completion date cannot be before target date."})
        

class OfficeExpence(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    date = models.DateField(null=True, blank=True)
    expence = models.CharField(max_length=255)
    debit = models.DecimalField(max_digits=10, decimal_places=2, blank=True, null=True)
    credit = models.DecimalField(max_digits=10, decimal_places=2, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"{self.expence} - {self.debit} on {self.credit}"
    
    class Meta:
        ordering = ['-created_at']


class ActivityStatus(enum.Enum):
    PENDING = 'pending'
    IN_PROGRESS = 'in_progress'
    COMPLETED = 'completed'
    CANCELLED = 'cancelled'


class Activity(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    title = models.CharField(max_length=255)
    description = models.TextField(blank=True, null=True)
    status = models.CharField(
        max_length=20, 
        choices=choices_the_options(ActivityStatus), 
        default=ActivityStatus.PENDING.value
    )
    assigned_to = models.ForeignKey(
        Profile, 
        on_delete=models.SET_NULL, 
        null=True, 
        blank=True, 
        related_name='assigned_activities'
    )
    created_by = models.ForeignKey(
        Profile, 
        on_delete=models.SET_NULL, 
        null=True, 
        blank=True, 
        related_name='created_activities'
    )
    due_date = models.DateTimeField(null=True, blank=True)
    completed_at = models.DateTimeField(null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return self.title
    
    class Meta:
        ordering = ['-created_at']
        verbose_name_plural = 'Activities'




